#include "pmbus_i2c.h"

int pmbus_setup()
{
	int Status;

	Status = ps_setup_i2c();
	if (Status != XST_SUCCESS)
	{
		xil_printf("ERROR: PS i2c setup failed \n");
		return Status;
	}

    Status = i2c_mux_reset();
    if(Status != XST_SUCCESS) {
        xil_printf("ERROR: Unable to reset i2c multiplexer\r\n");
        return Status;
    }

    Status = i2c_mux_setup();
    if(Status != XST_SUCCESS) {
        xil_printf("ERROR: Unable to configure i2c multiplexer\r\n");
        return Status;
    }
return XST_SUCCESS;
}

int ps_setup_i2c()
{
	int Status;
	XIicPs_Config *Config;

	Config = XIicPs_LookupConfig(I2C_DEVICE_ID);
	if (NULL == Config)
		return XST_FAILURE;

	Status = XIicPs_CfgInitialize(&i2c, Config, Config->BaseAddress);
	if (Status != XST_SUCCESS)
		return XST_FAILURE;

	Status = XIicPs_SelfTest(&i2c);
	if (Status != XST_SUCCESS)
		return XST_FAILURE;

	XIicPs_SetSClk(&i2c, I2C_SCLK_RATE);
	return XST_SUCCESS;
}

int i2c_mux_reset()
{
	XGpioPs_Config *ConfigPtr;
	XGpioPs Gpio;
	int Status;

	ConfigPtr = XGpioPs_LookupConfig(XPAR_GPIO_0_DEVICE_ID);
	Status = XGpioPs_CfgInitialize(&Gpio, ConfigPtr, ConfigPtr->BaseAddr);
	if (Status != XST_SUCCESS)
		return XST_FAILURE;

	XGpioPs_SetDirectionPin(&Gpio, I2C_MUX_NOT_RST_PIN, 1);
	XGpioPs_SetOutputEnablePin(&Gpio, I2C_MUX_NOT_RST_PIN, 1);

	// reset i2c multiplexer
	int iii=0;
	XGpioPs_WritePin(&Gpio, I2C_MUX_NOT_RST_PIN, 0x0);
	for(iii=0; iii<0x0FFFFFF; iii++);
	XGpioPs_WritePin(&Gpio, I2C_MUX_NOT_RST_PIN, 0x1);
	for(iii=0; iii<0x0FFFFFF; iii++);

	return XST_SUCCESS;
}

int i2c_mux_setup()
{
	int Status;
    unsigned char buffer = I2C_MUX_CHANNEL_MASK;
    // Setup i2c multiplexer channel (Channel 4 is connected to PMBus)
    while(XIicPs_BusIsBusy(&i2c));
    Status = XIicPs_MasterSendPolled(&i2c, &buffer, 1, I2C_MUX_ADDRESS);
    if(Status != XST_SUCCESS){
        return XST_FAILURE;}

    while(XIicPs_BusIsBusy(&i2c));
    Status = XIicPs_MasterRecvPolled(&i2c, &buffer, 1, I2C_MUX_ADDRESS);
    if(Status != XST_SUCCESS)
        return Status;

    if(buffer != I2C_MUX_CHANNEL_MASK) {
        xil_printf("ERROR: IIC multiplexer read back 0x%02X expected 0x%02X\r\n", buffer, I2C_MUX_CHANNEL_MASK);
        return XST_FAILURE;
    }

return XST_SUCCESS;

}

int pmbus_write(unsigned char address, unsigned char command, unsigned char data) {
    unsigned char writeBuffer[2];
    unsigned int status;

    writeBuffer[0] = command;
    writeBuffer[1] = data;


    while(XIicPs_BusIsBusy(&i2c));
    status =  XIicPs_MasterSendPolled(&i2c, writeBuffer, 2, address);
    if(status != XST_SUCCESS) {
        xil_printf("SEND ERROR: 0x%08X\r\n", status);
        return status;
    }

    return XST_SUCCESS;
}

int pmbus_read(unsigned char address, unsigned char command, unsigned char byteCount, unsigned char *buffer) {
    unsigned int status;

    while(XIicPs_BusIsBusy(&i2c));
    status = XIicPs_SetOptions(&i2c, XIICPS_REP_START_OPTION);
    if(status != XST_SUCCESS) {
        xil_printf("ERROR: Unable to set repeated start option: 0x%08X\r\n", status);
        return status;
    }

    status = XIicPs_MasterSendPolled(&i2c, &command, 1, address);
    if(status != XST_SUCCESS) {
        xil_printf("ERROR: RX send error: 0x%08X\r\n", status);
        return status;
    }

	int iii=0;
	for(iii=0; iii<0x0FFF; iii++);

    status = XIicPs_MasterRecvPolled(&i2c, buffer, byteCount, address);
    if(status != XST_SUCCESS) {
        status = XIicPs_ReadReg(i2c.Config.BaseAddress, XIICPS_ISR_OFFSET);
        xil_printf("ERROR: RX error: 0x%08X\r\n", status);
        return status;
    }

    status = XIicPs_ClearOptions(&i2c, XIICPS_REP_START_OPTION);
    if(status != XST_SUCCESS) {
        xil_printf("ERROR: Unable to clear repeated start option: 0x%08X\r\n", status);
        return status;
    }


    return XST_SUCCESS;
}

int check_Vout_Mode()
{
	unsigned char buffer;
    // Setup VOUT Resolution
    pmbus_write(I2C_SLAVE_ADDR, CMD_PAGE, 0x0);
    pmbus_read(I2C_SLAVE_ADDR, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;

    pmbus_write(I2C_SLAVE_ADDR+1, CMD_PAGE, 0x0);
    pmbus_read(I2C_SLAVE_ADDR+1, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;

    pmbus_write(I2C_SLAVE_ADDR, CMD_PAGE, 0x1);
    pmbus_read(I2C_SLAVE_ADDR, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;
    pmbus_write(I2C_SLAVE_ADDR+1, CMD_PAGE, 0x1);
    pmbus_read(I2C_SLAVE_ADDR+1, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;

    pmbus_write(I2C_SLAVE_ADDR, CMD_PAGE, 0x2);
    pmbus_read(I2C_SLAVE_ADDR, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;
    pmbus_write(I2C_SLAVE_ADDR+1, CMD_PAGE, 0x2);
    pmbus_read(I2C_SLAVE_ADDR+1, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;

    pmbus_write(I2C_SLAVE_ADDR, CMD_PAGE, 0x3);
    pmbus_read(I2C_SLAVE_ADDR, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;
    pmbus_write(I2C_SLAVE_ADDR+1, CMD_PAGE, 0x3);
    pmbus_read(I2C_SLAVE_ADDR+1, CMD_VOUT_MODE, 1, &buffer);
    if(buffer != 0x18)
    	return 1;
return 0;
}

#define CMD_READ_VOUT       0x8B
#define CMD_READ_IOUT       0x8C
#define CMD_READ_TEMP       0x8D
#define CMD_READ_POUT       0x96

#define EPSILON       100
double Measure_Sensors(double pre_total_pow)
{
	int page;
	double voltage;
	double current;
	double pow [10];
	int temp;
	double total_pow = 0;

	for(page = 0; page < 5; page++){
    pmbus_write(I2C_SLAVE_ADDR, CMD_PAGE, page);
    //voltage = read_voltage(I2C_SLAVE_ADDR)/1000;
    //current = read_current(I2C_SLAVE_ADDR);
    //temp = read_temp(I2C_SLAVE_ADDR);
    pow[page] = read_power(I2C_SLAVE_ADDR);
    //printf("Page%d Bank1 => vol: %f V, cur: %f mA, pow: %f(%f) mW, Temp: %d c\n", page, voltage, current, pow, voltage*current, temp);

    pmbus_write(I2C_SLAVE_ADDR+1, CMD_PAGE, page);
    //voltage = read_voltage(I2C_SLAVE_ADDR+1)/1000;
    //current = read_current(I2C_SLAVE_ADDR+1);
    //temp = read_temp(I2C_SLAVE_ADDR+1);
    pow[5+page] = read_power(I2C_SLAVE_ADDR+1);
    total_pow += pow[page] + pow[5+page];
    //printf("Page%d Bank2 => vol: %f V, cur: %f mA, pow: %f(%f) mW, Temp: %d c\n", page, voltage, current, pow, voltage*current, temp);
	}
	if ((total_pow > (pre_total_pow + EPSILON)) || (total_pow < (pre_total_pow - EPSILON))){
		printf("pow: %f W (%f, %f, %f, %f, %f, %f, %f, %f, %f, %f) mW\n", total_pow/1000, pow[0], pow[1], pow[2], pow[3], pow[4], pow[5], pow[6], pow[7], pow[8], pow[9]);
		return total_pow;
	}
	return pre_total_pow;
}

double read_voltage(int Addr)
{
	unsigned char buffer[2];
    u16 data;
    double voltage;

    pmbus_read(Addr, CMD_READ_VOUT, 2, buffer);
    data = (u16)((buffer[0]) | (buffer[1]) << 8);
    voltage = (double)data*MODE_8_VALUE;
    return voltage;
}

double read_current(int Addr)
{
    double current;
    unsigned char buffer[2];
    u16 data;
    pmbus_read(Addr, CMD_READ_IOUT, 2, buffer);
	data = (u16)((buffer[0]) | (buffer[1]) << 8);


	signed char exponent = data >> 11;
	signed short mantissa = data & 0x7ff;
	if(mantissa & 0x400){
		xil_printf("negative current??\n");
		return -1;}

	if(exponent == MODE_6_EXP)
		current = mantissa*MODE_6_VALUE;
	else if(exponent == MODE_3_EXP)
		current = mantissa*MODE_3_VALUE;
	else if(exponent == MODE_10_EXP)
		current = mantissa*MODE_10_VALUE;
	else
		current = -1;

	return current;
}

int read_temp(int Addr)
{
    int temp;
    unsigned char buffer[2];
    u16 data;
    pmbus_read(Addr, CMD_READ_TEMP, 2, buffer);
	data = (u16)((buffer[0]) | (buffer[1]) << 8);


	signed char exponent = data >> 11;
	signed short mantissa = data & 0x7ff;
	if(mantissa & 0x400)
		temp = 0xFFFFF8 | mantissa;
	else
		temp = mantissa;
	if(exponent != 0)
		xil_printf("wrong Temperature??\n");

	return temp;
}

double read_power(int Addr)
{
    double pow;
    unsigned char buffer[2];
    u16 data;
    pmbus_read(Addr, CMD_READ_POUT, 2, buffer);
	data = (u16)((buffer[0]) | (buffer[1]) << 8);


	signed char exponent = data >> 11;
	signed short mantissa = data & 0x7ff;
	if(mantissa & 0x400){
		xil_printf("negative power??\n");
		return -1;}

	if(exponent == MODE_5_EXP)
		pow = mantissa*MODE_5_VALUE;
	else if(exponent == MODE_2_EXP)
		pow = mantissa*MODE_2_VALUE;
	else if(exponent == MODE_6_EXP)
		pow = mantissa*MODE_6_VALUE;
	else
		pow = -1;

	return pow;
}
