

typedef struct {
  short int re;
  short int im;
} complex;

#define IS_HLS
#ifdef IS_HLS
#define PRAGMA_SUB(x) _Pragma (#x)
#define DO_PRAGMA(x) PRAGMA_SUB(x)
#endif

#define NMB_SLOT        2
#define OFDM_IN_SLOT    2
#define RX_ANT	        4
#define MAX_LAYERS	  	4
#define MAX_SC			1200

typedef enum
{
	MOD_PSK = 1,
	MOD_QPSK = 2,
	MOD_16QAM = 4,
	MOD_64QAM = 6,
	MOD_256QAM = 8
} mod_type;

complex cmul(complex a, complex b)
{
	complex t;
	t.re = a.re * b.re - a.im * b.im;
	t.im = a.im * b.re + a.re * b.im;
	return t;
}

int mod255(unsigned int  a)
{
    a = (a >> 16) + (a & 0xFFFF); // sum base 2**16 digits
    a = (a >>  8) + (a & 0xFF);   // sum base 2**8  digits
    if (a < 255)
    	return a;
    if (a < (2 * 255))
    	return a - 255;
    return a - (2 * 255);
}

void soft_demap(
		complex in[2*(OFDM_IN_SLOT-1)*MAX_SC*MAX_LAYERS],
		int scaling_factor,
		int mod,
		int n,
		char out[2*2*(OFDM_IN_SLOT-1)*MAX_SC*MOD_256QAM*MAX_LAYERS],
		char* err) {
#ifdef IS_HLS
#pragma HLS DATA_PACK variable=in
#pragma HLS RESOURCE variable=in core=RAM_1P_BRAM
#pragma HLS RESOURCE variable=out core=RAM_1P_BRAM
#endif

int L;
int i,k;
*err=0;
switch (mod)
	{
	case MOD_PSK:
		{
		int ind = 0;
		for (i=0; i<n; i++)
			{
			#ifdef IS_HLS
			#pragma HLS PIPELINE
			DO_PRAGMA(HLS LOOP_TRIPCOUNT max=48*MAX_SC)
			#endif
			out[ind++] = in[i].re * scaling_factor;
			}
		}
		break;
	case MOD_QPSK:
		{
		int ind = 0;
		complex exp_pi4, temp;
		exp_pi4.re = 4096;
		exp_pi4.im = -4096;
		for (i=0; i<n; i++)
			{
			#ifdef IS_HLS
			#pragma HLS PIPELINE
			DO_PRAGMA(HLS LOOP_TRIPCOUNT max=48*MAX_SC)
			#endif
			temp = cmul(in[i], exp_pi4);
			out[ind++] = mod255((temp.re * scaling_factor));//%255;
			out[ind++] = mod255((temp.im * scaling_factor));//%255;
			}
		}
		break;
	case MOD_16QAM:
		{
		int temp_real, temp_imag;
		int ind = 0;

		for (i=0; i<n; i++)
			{
			#ifdef IS_HLS
			DO_PRAGMA(HLS LOOP_TRIPCOUNT max=48*MAX_SC)
			#endif
			for (k=0; k<4; k++)
				{
				#ifdef IS_HLS
				#pragma HLS PIPELINE
				#endif
				L = k*4096;
				temp_real = (int)(((L - 1) - ((in[i].re * scaling_factor)>>8) + (L - 1)) / 2);
				temp_imag = (int)(((L - 1) - ((in[i].im * scaling_factor)>>8) + (L - 1)) / 2);
				if (temp_real < 0)
					temp_real = 0;
				else if (temp_real > (L - 1))
					temp_real = (L - 1);
				if (temp_imag < 0)
					temp_imag = 0;
				else if (temp_imag > (L - 1))
					temp_imag = (L - 1);
				out[ind++] = mod255(temp_real);//%255;
				out[ind++] = mod255(temp_imag);//%255;
				}
			}
		}
		break;
	case MOD_64QAM:
		{
		int temp_real, temp_imag;
		int ind = 0;
		for (i=0; i<n; i++)
			{
			#ifdef IS_HLS
			DO_PRAGMA(HLS LOOP_TRIPCOUNT max=48*MAX_SC)
			#endif
			for (k=0; k<6; k++)
				{
				#ifdef IS_HLS
				#pragma HLS PIPELINE
				#endif
				L = k*4096;
				temp_real = (int)(((L - 1) - ((in[i].re * scaling_factor)>>8) + (L - 1)) / 2);
				temp_imag = (int)(((L - 1) - ((in[i].im * scaling_factor)>>8) + (L - 1)) / 2);
				if (temp_real < 0)
					temp_real = 0;
				else if (temp_real > (L - 1))
					temp_real = (L - 1);
				if (temp_imag < 0)
					temp_imag = 0;
				else if (temp_imag > (L - 1))
					temp_imag = (L - 1);
				out[ind++] = mod255(temp_real);//%255;
				out[ind++] = mod255(temp_imag);//%255;
				}
			}
		}
		break;
	  case MOD_256QAM:
	    {
	      int temp_real, temp_imag;
	      int ind = 0;
	      for (i=0; i<n; i++) {
			#ifdef IS_HLS
			DO_PRAGMA(HLS LOOP_TRIPCOUNT max=48*MAX_SC)
			#endif
			for (k=0; k<8; k++) {
				#ifdef IS_HLS
				#pragma HLS PIPELINE
				#endif
		  		L = k*4096;
		  		temp_real = (int)(((L - 1) - ((in[i].re * scaling_factor)>>8) + (L - 1)) / 2);
		  		temp_imag = (int)(((L - 1) - ((in[i].im * scaling_factor)>>8) + (L - 1)) / 2);
		  		if (temp_real < 0)
		  		  temp_real = 0;
		  		else if (temp_real > (L - 1))
		  		  temp_real = (L - 1);
		  		if (temp_imag < 0)
		  		  temp_imag = 0;
		  		else if (temp_imag > (L - 1))
		  		  temp_imag = (L - 1);
				out[ind++] = mod255(temp_real);//%255;
				out[ind++] = mod255(temp_imag);//%255;
			}
	      }
	    }
	    break;
	default:
		*err=1;
	}
}
